<?php
/**
 * Plugin Name: TSV Product Feed for FluentCart
 * Plugin URI: https://webv8.net/
 * Description: Generates a TSV product feed (including variations) from FluentCart products for Google Merchant Center and similar catalog tools. Provides a simple /gmc-feed.tsv endpoint and a Tools page to copy the feed URL.
 * Version: 1.8
 * Author: Web V8
 * Author URI: https://webv8.net/
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) { exit; }

define('V8_FC_TSV_FEED_VERSION', '1.8');
define('V8_FC_TSV_FEED_QUERY_VAR', 'v8_gmc_feed');
define('V8_FC_TSV_FEED_SLUG', 'gmc-feed.tsv');

function v8_fc_tsv_feed_register_rewrite_rule() {
	add_rewrite_rule('^' . preg_quote(V8_FC_TSV_FEED_SLUG, '/') . '$', 'index.php?' . V8_FC_TSV_FEED_QUERY_VAR . '=1', 'top');
}
add_action('init', 'v8_fc_tsv_feed_register_rewrite_rule');

function v8_fc_tsv_feed_add_query_var($vars) {
	$vars[] = V8_FC_TSV_FEED_QUERY_VAR;
	return $vars;
}
add_filter('query_vars', 'v8_fc_tsv_feed_add_query_var');

function v8_fc_tsv_feed_activate() {
	v8_fc_tsv_feed_register_rewrite_rule();
	flush_rewrite_rules(false);
}
register_activation_hook(__FILE__, 'v8_fc_tsv_feed_activate');

function v8_fc_tsv_feed_deactivate() {
	flush_rewrite_rules(false);
}
register_deactivation_hook(__FILE__, 'v8_fc_tsv_feed_deactivate');

function v8_fc_tsv_feed_get_feed_url() {
	return home_url('/' . V8_FC_TSV_FEED_SLUG);
}

function v8_fc_tsv_feed_template_redirect() {
	if ((int) get_query_var(V8_FC_TSV_FEED_QUERY_VAR) !== 1) {
		return;
	}

	// Headers
	header('Content-Type: text/tab-separated-values; charset=UTF-8');
	header('Content-Disposition: inline; filename=gmc-feed.tsv');
	header('X-Content-Type-Options: nosniff');
	header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
	header('Pragma: no-cache');
	header('Expires: 0');

	$out = fopen('php://output', 'w');
	if (!$out) {
		status_header(500);
		exit;
	}

	$headers = [
		'id',
		'item_group_id',
		'title',
		'description',
		'link',
		'image_link',
		'availability',
		'price',
		'condition',
		'brand',
		'color'
	];
	fputcsv($out, $headers, "\t");

	$products = get_posts([
		'post_type'      => 'fluent-products',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
		'fields'         => 'ids'
	]);

	foreach ($products as $post_id) {
		$post_id = (int) $post_id;

		$title       = get_the_title($post_id);
		$description = wp_strip_all_tags(get_the_excerpt($post_id));
		if ($description === '') {
			$description = wp_strip_all_tags(get_post_field('post_content', $post_id));
		}

		$link = get_permalink($post_id);

		// Product main image
		$image_link = '';
		$thumb_id = (int) get_post_thumbnail_id($post_id);
		if ($thumb_id) {
			$image_link = wp_get_attachment_url($thumb_id) ?: '';
		}

		// Brand taxonomy
		$brand = '';
		$brand_terms = wp_get_post_terms($post_id, 'product-brands', ['fields' => 'names']);
		if (!is_wp_error($brand_terms) && !empty($brand_terms)) {
			$brand = (string) $brand_terms[0];
		}

		$condition = 'new';

		// Variations (via FluentCart model)
		$variations = [];
		if (class_exists('\FluentCart\App\Models\ProductVariation')) {
			try {
				$variations = \FluentCart\App\Models\ProductVariation::where('post_id', $post_id)->get();
			} catch (\Throwable $e) {
				$variations = [];
			}
		}

		/**
		 * Suppress fake variation:
		 * If there is exactly ONE variation and its variation_title == product title (case-insensitive),
		 * treat it like a simple product row.
		 */
		$has_fake_single_variation = false;
		if (!empty($variations) && is_countable($variations) && count($variations) === 1) {
			$only = $variations[0];
			$only_title = trim((string) ($only->variation_title ?? ''));
			if ($only_title !== '' && strcasecmp($only_title, $title) === 0) {
				$has_fake_single_variation = true;
			}
		}

		// If product has variations and NOT a fake single variation: output one row per variation
		if (!empty($variations) && !$has_fake_single_variation) {
			foreach ($variations as $v) {
				$variation_id = (int) ($v->id ?? 0);
				if (!$variation_id) {
					continue;
				}

				$g_id          = $post_id . '-' . $variation_id;
				$item_group_id = (string) $post_id;

				$color = trim((string) ($v->variation_title ?? ''));

				$row_title = $title;
				if ($color !== '') {
					$row_title .= ' - ' . $color;
				}

				// Always use product page as link (best for GMC)
				$row_link = $link;

				$row_image = $image_link;
				if (isset($v->thumbnail) && $v->thumbnail) {
					$row_image = (string) $v->thumbnail;
				}

				$stock_status = strtolower((string) ($v->stock_status ?? ''));
				$availability = ($stock_status === 'out_of_stock' || $stock_status === 'outofstock' || $stock_status === '0')
					? 'out of stock'
					: 'in stock';

				$raw_price = $v->item_price ?? 0;
				$raw_price = is_numeric($raw_price) ? (float) $raw_price : 0.0;
				$price_value = ($raw_price >= 1000) ? ($raw_price / 100) : $raw_price;
				$price = number_format($price_value, 2, '.', '') . ' AUD';

				fputcsv($out, [
					$g_id,
					$item_group_id,
					$row_title,
					$description,
					$row_link,
					$row_image,
					$availability,
					$price,
					$condition,
					$brand,
					$color
				], "\t");
			}

			continue;
		}

		/**
		 * Simple row output:
		 * - Either no variations, OR it was a fake single variation.
		 * We still try to pull price/availability from the single variation if it exists (more reliable).
		 */
		$availability = 'in stock';
		$price = '';

		if (!empty($variations) && $has_fake_single_variation) {
			$v = $variations[0];

			$stock_status = strtolower((string) ($v->stock_status ?? ''));
			$availability = ($stock_status === 'out_of_stock' || $stock_status === 'outofstock' || $stock_status === '0')
				? 'out of stock'
				: 'in stock';

			$raw_price = $v->item_price ?? 0;
			$raw_price = is_numeric($raw_price) ? (float) $raw_price : 0.0;
			$price_value = ($raw_price >= 1000) ? ($raw_price / 100) : $raw_price;
			$price = number_format($price_value, 2, '.', '') . ' AUD';

			// Use variation thumbnail if available
			if (isset($v->thumbnail) && $v->thumbnail) {
				$image_link = (string) $v->thumbnail;
			}
		}

		fputcsv($out, [
			(string) $post_id,  // id
			'',                 // item_group_id
			$title,             // title
			$description,
			$link,
			$image_link,
			$availability,
			$price,
			$condition,
			$brand,
			''                  // color
		], "\t");
	}

	fclose($out);
	exit;
}
add_action('template_redirect', 'v8_fc_tsv_feed_template_redirect');

/**
 * Tools page: shows the feed URL for copy/paste
 */
function v8_fc_tsv_feed_register_tools_page() {
	add_management_page(
		'TSV Product Feed',
		'TSV Product Feed',
		'manage_options',
		'v8-fc-tsv-feed',
		'v8_fc_tsv_feed_render_tools_page'
	);
}

// Add "Settings" link on Plugins page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
	$settings_link = '<a href="' . esc_url(admin_url('tools.php?page=v8-fc-tsv-feed')) . '">Settings</a>';
	array_unshift($links, $settings_link);
	return $links;
});

add_action('admin_menu', 'v8_fc_tsv_feed_register_tools_page');

function v8_fc_tsv_feed_render_tools_page() {
	if (!current_user_can('manage_options')) {
		return;
	}

	$feed_url = v8_fc_tsv_feed_get_feed_url();
	?>
	<div class="wrap">
		<h1>TSV Product Feed</h1>
		<p>Copy this TSV feed URL into Google Merchant Center (or any catalog tool that accepts TSV feeds).</p>

		<table class="form-table" role="presentation">
			<tbody>
				<tr>
					<th scope="row"><label for="v8_fc_tsv_feed_url">Feed URL</label></th>
					<td>
						<input type="text" id="v8_fc_tsv_feed_url" class="regular-text" value="<?php echo esc_url($feed_url); ?>" readonly>
						<button type="button" class="button" id="v8_fc_tsv_feed_copy_btn">Copy</button>
						<p class="description">This endpoint is generated by the plugin at /gmc-feed.tsv</p>
					</td>
				</tr>
			</tbody>
		</table>

		<script>
		(function(){
			var btn = document.getElementById('v8_fc_tsv_feed_copy_btn');
			var input = document.getElementById('v8_fc_tsv_feed_url');
			if(!btn || !input) return;

			btn.addEventListener('click', async function(){
				input.focus();
				input.select();
				try {
					if (navigator.clipboard && navigator.clipboard.writeText) {
						await navigator.clipboard.writeText(input.value);
					} else {
						document.execCommand('copy');
					}
					btn.textContent = 'Copied';
					setTimeout(function(){ btn.textContent = 'Copy'; }, 1200);
				} catch(e) {
					btn.textContent = 'Copy failed';
					setTimeout(function(){ btn.textContent = 'Copy'; }, 1500);
				}
			});
		})();
		</script>
	</div>
	<?php
}
